(***************************************************************************
 This file is Copyright (C) 2007 Christoph Reichenbach

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the
   Free Software Foundation, Inc.
   59 Temple Place, Suite 330
   Boston, MA  02111-1307
   USA

 The author can be reached as "reichenb" at "colorado.edu".

***************************************************************************)

structure HILT =
struct

val version = "hilt, v0.1"
val copyright = "This program is Copyright (C) 2007 Christoph Reichenbach."

datatype colour_attr = FOREGROUND	of char
		     | BACKGROUND	of char
		     | BOLD
		     | UNDERLINE

val colours = [("black", #"0"),
	       ("red", #"1"),
	       ("green", #"2"),
	       ("yellow", #"3"),
	       ("blue", #"4"),
	       ("magenta", #"5"),
	       ("cyan", #"6"),
	       ("white", #"7")]

fun ansi_set_graphics (s) = "\027[" ^ s ^ "m"

fun ansi_setmode (s) =
    let fun embed_mode (BOLD)		= "1"
	  | embed_mode (UNDERLINE)	= "4"
	  | embed_mode (FOREGROUND c)	= "3" ^ (implode [c])
	  | embed_mode (BACKGROUND c)	= "4" ^ (implode [c])
    in ansi_set_graphics (String.concatWith ";" (map embed_mode s))
    end

fun get_highlighting_by_nr (n) =
    let fun get k v = (v mod k, v div k)
	fun getb v = case get 2 v of
			 (0, n)	=> (false, n)
		       | (1, n)	=> (true, n)
		       | (_, _)	=> raise Fail "n mod 2 gave something other than 0 or 1"
	val (fg, n) = get 7 n
	val (unbold, n) = getb n
	val bold = not unbold
	val (bg, n) = get 8 n
	val (underline, n) = getb n
	val fg = if fg >= bg
		 then fg + 1
		 else fg
	val props = [FOREGROUND (chr (fg + (ord #"0")))]
	val props = if bold
		    then BOLD :: props
		    else props
	val props = if underline
		    then UNDERLINE :: props
		    else props
	val props = if (bg > 0)
		    then (BACKGROUND (chr (bg + (ord #"0")))) :: props
		    else props
    in ansi_setmode (props)
    end

val ansi_reset = ansi_set_graphics ("0")

structure RE = RegExpFn(structure P = AwkSyntax
			structure E = DfaEngine)

val re = RE.compileString("x[^x]*x")
val re = RE.compileString("1")
val input = TextIO.StreamIO.input1

val reader = RE.prefix re input

type pat_spec = { pattern	: RE.regexp,
		  prefix	: string,
		  suffix	: string }

local
    fun list_reader ([])	= NONE
      | list_reader (h::tl)	= SOME (h, tl)

in
    fun mark_input (all_patterns : pat_spec list) (outstream) (instream) =
	let val instream' = TextIO.getInstream instream
	    fun putc (c) = TextIO.output1 (outstream, c)
	    fun printl []	= ()
	      | printl (h::tl)	= (putc h;
				   printl tl)
	    val print = printl o explode

	    fun try_match ([]) ([])	= ()
	      | try_match ([]) (h::tl)	= (putc (h);
					   try_match (all_patterns) (tl))
	      | try_match ({ pattern, prefix, suffix } :: patterns)
			  (str) =
		case RE.prefix (pattern) (list_reader) (str) of
		    NONE		=> try_match (patterns) (str)
		  | SOME (result,
			  strtail)	=> (case MatchTree.root result of
						NONE		=> raise Fail "Unexpected nonmatch"
					      | SOME { len,
						       pos }	=> (print (prefix);
								    printl (List.take (pos, len));
								    print (suffix);
								    try_match (all_patterns) (strtail)))

	    fun readl (ins) =
		case (TextIO.StreamIO.inputLine (ins)) of
		    NONE	=> ()
		  | SOME (str,
			  ins)	=> (try_match (all_patterns) (explode str);
				    readl (ins))
	in readl (instream')
	end
end

structure GO = GetOpt

fun main (selfname, args) =
    let val help = ref false
	val output = ref TextIO.stdOut
	val inputs = ref []
	val options = [{ desc	= GO.NoArg (fn () => help := true),
                         help	= "Print usage information (this message) and exit",
                         long	= ["help"],
                         short	= "h" },

                       { desc	= GO.NoArg (fn () => (print version; print "\n"; print copyright; print "\n")),
			 help	= "Print the current program version and copyright, then exit",
			 long	= ["version"],
			 short	= "V" },

		       { desc	= GO.ReqArg ((fn s => inputs := (s :: (!inputs))), "input file"),
			 help	= "Read input from a file (rather than from stdin)",
			 long	= ["input"],
			 short	= "f"},

		       { desc	= GO.ReqArg ((fn s => output := TextIO.openOut (s)), "output file"),
			 help	= "Write output to a file (rather than to stdout)",
			 long	= ["output"],
			 short	= "o" }]

	fun perr (s) = TextIO.output (TextIO.stdErr, s)

        fun print_help () =
            (print (GO.usageInfo    { header        = selfname ^ " [options]* [regexps]+\n",
                                      options       = options });
             print "\n")

        val (_,
             reststrings) = GO.getOpt ({ argOrder   = GO.Permute,
                                         errFn      = (fn s =>
                                                          (perr s;
                                                           perr "\n";
                                                           OS.Process.exit OS.Process.failure)),
                                         options    = options })
                                      args

	val () = if !help
		 then (print_help ();
		       OS.Process.exit OS.Process.success)
		 else ()

	fun regexp (n, [])	= []
	  | regexp (n, h::tl)	= { prefix	= (get_highlighting_by_nr (n)),
				    pattern	= RE.compileString h,
				    suffix	= ansi_reset } :: regexp (n+1, tl)

	val markfn = mark_input (regexp (0, reststrings)) (!output)
	fun markfn_file filename = let val file = TextIO.openIn filename
				       val () = markfn file
				   in TextIO.closeIn file
				   end 

    in (if reststrings = []
	then (print_help ();
	      OS.Process.failure)
	else ((if null (!inputs)
	       then markfn TextIO.stdIn
	       else app markfn_file (rev (!inputs)));
	      OS.Process.success))
       before TextIO.closeOut (!output)
    end

(*
val () = mark_input [{ prefix = ansi_setmode [BOLD, UNDERLINE, FOREGROUND #"1" ],
		       pattern = RE.compileString ("X"),
		       suffix = ansi_reset },
		     { prefix = "{",
		       pattern = RE.compileString ("[0-9]+"),
		       suffix = "}" }] TextIO.stdOut TextIO.stdIn
 *)

val () = OS.Process.exit (main (CommandLine.name (), CommandLine.arguments ()))

end
