(***************************************************************************
 This file is Copyright (C) 2005 Christoph Reichenbach

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the
   Free Software Foundation, Inc.
   59 Temple Place, Suite 330
   Boston, MA  02111-1307
   USA

 The author can be reached as "reichenb" at "colorado.edu".

***************************************************************************)

structure Readline_Base =
struct
      datatype parse_error =
	       INSUFFICIENT_ARGUMENTS
	     | IMPROPER_ARGUMENT of (int * string (* typename *) * string (* bad argument *) )
	     | TOO_MANY_ARGUMENTS of (int (* expected *) * int (* additional *))

      exception ParseError of parse_error	(* Failed to represent token into given structure *)

      type 'a rlfun_ctx = 'a ReadlineFoundation.rlfun_ctx
      type rlfun = ReadlineFoundation.rlfun

      fun rlfun (name : string, f) = { name		= name,
				       f		= fn params => (params, 1, f),
				       completions	= [] }

      fun apply_argument (reader : string -> 'a option,
			  acceptor : string list -> (string list * int * ('a -> 'b)),
			  typename : string) =
	  fn stringlist => let val (stringlist_rest, index, f') = acceptor stringlist
			   in case stringlist_rest of
				  []	=> raise ParseError INSUFFICIENT_ARGUMENTS
				| h::t	=> case (reader h) of
					       NONE	=> raise ParseError (IMPROPER_ARGUMENT (index, typename, h))
					     | SOME v	=> (t, index + 1, f' v)	(* Success *)
			   end

      fun apply_const (const : 'a)
		      ({ name		= name,
			 f		= f,
			 completions	= completions } : ('a -> 'b) rlfun_ctx) =
		      { name	= name,
			f	= fn stringlist => let val (rest, index, f') = f stringlist
						   in (rest, index, f' const)
						   end,
			completions = completions }

      fun new_rlfun (prectx : 'a rlfun_ctx,
		     printer : 'a -> unit) =
	  { name	= #name prectx,
	    completions	= #completions prectx,
	    function	= fn stringlist =>
			     let val (rest, index, f) = (#f prectx) stringlist
			     in case rest of
				    []	=> printer f
				  | l	=> raise ParseError (TOO_MANY_ARGUMENTS (index - 1, length l))
			     end }

      fun stringify_index 0	= "zeroeth"
	| stringify_index 1	= "first"
	| stringify_index 2	= "second"
	| stringify_index 3	= "third"
	| stringify_index n	= case (n mod 10) of
				      1	=> (Int.toString n) ^ "st"
				    | 2	=> (Int.toString n) ^ "nd"
				    | 3	=> (Int.toString n) ^ "rd"
				    | _	=> (Int.toString n) ^ "th"

      fun stringify_error INSUFFICIENT_ARGUMENTS		= "Insufficient number of arguments"
	| stringify_error (IMPROPER_ARGUMENT (idx, tn, arg))	= "Improper argument for "
								  ^ (stringify_index idx)
								  ^ " parameter: Expected "
								  ^ tn
								  ^ ", got '"
								  ^ arg
								  ^ "'"
	| stringify_error (TOO_MANY_ARGUMENTS (max, add))	= "Too many arguments: Expected "
								  ^ (Int.toString max)
								  ^ ", encountered "
								  ^ (Int.toString (max + add))
end

functor RLMakeApp (structure RLAppType : READLINE_APP_TYPE) :> READLINE_APP
  where App = RLAppType =
struct
	structure App = RLAppType
	type t = App.t

	type 'a rlfun_ctx = 'a Readline_Base.rlfun_ctx
	type rlfun = Readline_Base.rlfun

	fun app (ctx : (App.t -> 'a) rlfun_ctx) =
	    { name		= #name ctx,
	      f			= Readline_Base.apply_argument (App.read,
								(#f ctx),
								App.typename),
	      completions	= App.completions :: (#completions ctx) }

end

functor RLMakeResult (structure RLResultType : READLINE_RESULT_TYPE) :> READLINE_RESULT
  where Result = RLResultType =
struct
      structure Result = RLResultType
      type t = Result.t

      type 'a rlfun_ctx = 'a Readline_Base.rlfun_ctx
      type rlfun = Readline_Base.rlfun

      fun rlfunc (ctx : t rlfun_ctx) : rlfun =
	  Readline_Base.new_rlfun (ctx,
				   Result.print)
end

functor RLMakeAppResult (structure RLAppType : READLINE_APP_TYPE
			 val print : RLAppType.t -> unit) :> READLINE_APP_RESULT
  where App = RLAppType
    and type Result.t = RLAppType.t =
struct
      structure App = RLAppType
      structure Result = struct
			       type t = App.t
			       val print = print
			 end

      structure RL_App = RLMakeApp (structure RLAppType = App)
      structure RL_Result = RLMakeResult (structure RLResultType = Result)

      type t = Result.t

      type 'a rlfun_ctx = 'a Readline_Base.rlfun_ctx
      type rlfun = Readline_Base.rlfun

      val app = RL_App.app
      val rlfunc = RL_Result.rlfunc
end