(***************************************************************************
 This file is Copyright (C) 2005 Christoph Reichenbach

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the
   Free Software Foundation, Inc.
   59 Temple Place, Suite 330
   Boston, MA  02111-1307
   USA

 The author can be reached as "reichenb" at "colorado.edu".

***************************************************************************)

structure Readline_Glue =
struct
      structure C = C
      structure ZString = ZString

      datatype completion =
	       NONE
	     | ENUM of string list * completion
	     | ENUM_SPLIT of ((string * completion) list * completion)
	     | STRING of completion
	     | INT of completion

      val readline_p = F_readline.f

      fun readline (prompt) =
	  let
              val prompt' = ZString.dupML' prompt

	      val input' = (C.call' F_readline.typ
				    (C.U.fcast (C.Light.fptr (F_readline.fptr ())),
				     (prompt')))

	      val _ = if C.Ptr.isNull' input'
		      then ()
		      else C.call' F_add_history.typ
				   (C.U.fcast (C.Light.fptr (F_add_history.fptr())),
				    input')
	  in
	      ZString.toML' input'
              before C.free' prompt'
	  end


      fun init (progname : string) =
	  let
	      val progname' = ZString.dupML' progname
	  in
	      C.call' F_initialise_readline.typ
	      (C.U.fcast (C.Light.fptr (F_initialise_readline.fptr())),
	       progname')
	  end

      val table_type_regular = C.Get.sint' (G_table_type_regular.obj'())
      val table_type_number = C.Get.sint' (G_table_type_regular.obj'())
      val table_type_string = C.Get.sint' (G_table_type_regular.obj'())

      val rl_allocd_objs : C.voidptr list ref = ref []

      fun rl_table_free () =
	  let val _ = List.app C.free' (!rl_allocd_objs)
	      val _ = rl_allocd_objs := []
	  in ()
	  end

      fun rl_table_dupstring (s : string) =
	  let val s' = ZString.dupML' s
	      val _ = rl_allocd_objs := (C.Ptr.inject' s')::(!rl_allocd_objs)
	  in
	      s'
	  end

      fun rl_table_newtable (typ, next, entries_nr, entries) =
	  let val t = C.new' S__completion_table.size
	      val _ = rl_allocd_objs := (C.Ptr.inject' (C.Ptr.|&! t))::(!rl_allocd_objs)

	      val _ = C.Set.sint' (S__completion_table.f_type' t,
				   typ)
	      val _ = C.Set.ptr' (S__completion_table.f_next_table' t,
				  next)
	      val _ = C.Set.sint' (S__completion_table.f_entries_nr' t,
				   entries_nr)
	      val _ = C.Set.ptr' (S__completion_table.f_entries' t,
				  entries)
	  in
	      C.Ptr.|&! t
	  end

      fun rl_table_compentry (entries) =
	  let val earr = C.alloc' S__completion_entry.size (Word.fromInt (length entries))
	      val _ = rl_allocd_objs := (C.Ptr.inject' earr)::(!rl_allocd_objs)

	      fun init_entry (e, (stringobj, nextobj)) =
		  let
		      val _ = C.Set.ptr' (S__completion_entry.f_string' e,
					  stringobj)
		      val _ = C.Set.ptr' (S__completion_entry.f_next_table' e,
					  nextobj)
		  in
		      ()
		  end

	      val sub = C.Ptr.sub' S__completion_entry.size

	      fun make_entries ([], _, arr) = ()
		| make_entries (h::t, n, arr) = init_entry (sub (arr, n),
							    h)
						before make_entries (t, n+1, arr)

	      val _ = make_entries (entries, 0, earr) (* Set all entries *)
	  in
	      earr
	  end

      fun set_readline_table (completion : completion) =
	  let
	      val _ = rl_table_free () (* De-allocate old memory *)

	      fun mkc NONE		= C.Ptr.null'
		| mkc (INT comp)	= rl_table_newtable (table_type_number,
							     mkc comp,
							     0,
							     C.Ptr.null')
		| mkc (STRING comp)	= rl_table_newtable (table_type_string,
							     mkc comp,
							     0,
							     C.Ptr.null')
		| mkc (ENUM (strs, cm))	= let val comp' = mkc cm
					      val len = length strs
					  in
					      rl_table_newtable (table_type_regular,
								 comp',
								 Int32.fromInt len,
								 rl_table_compentry
								     (map
									  (fn s =>
									      (rl_table_dupstring s,
									       comp'))
									  strs))
					  end
		| mkc (ENUM_SPLIT
			   (cms, dflt))	= let val dflt_comp' = mkc dflt
					      val len = length cms
					  in
					      rl_table_newtable (table_type_regular,
								 dflt_comp',
								 Int32.fromInt len,
								 rl_table_compentry
								     (map
									  (fn (s, comp) =>
									      (rl_table_dupstring s,
									       mkc comp))
									  cms))
					  end

	      val completion' = mkc completion
	  in
	      C.call' F_set_readline_table.typ
	      (C.U.fcast (C.Light.fptr (F_set_readline_table.fptr())),
	       completion')
	  end

(*
      val test_glue = ENUM_SPLIT ([("varfun",
				    ENUM (["var", "x", "zeta"], NONE)),
				   ("bar", NONE),
				   ("quux", STRING (ENUM (["qx1","qx2"], NONE)))], NONE)
      val _ = init ".readline-glue-test."
      val _ = set_readline_table (test_glue)
      val _ = readline "> ";
*)
end