(***************************************************************************
 This file is Copyright (C) 2005 Christoph Reichenbach

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the
   Free Software Foundation, Inc.
   59 Temple Place, Suite 330
   Boston, MA  02111-1307
   USA

 The author can be reached as "reichenb" at "colorado.edu".

***************************************************************************)

functor Readline (val progname : string (* Program name, used for 'history' cache *)) :> READLINE =
struct
      structure RLB = Readline_Base
      structure RLG = Readline_Glue
      open RLB

      datatype menu_item = FUNCTION of RLB.rlfun
			 | SUBMENU of (string * menu)
      withtype menu = menu_item list

      type prompt = string

      fun menu_item (f : RLB.rlfun) : menu_item = FUNCTION f

      fun sub_menu (name : string,
		    items : menu) = SUBMENU (name, items)

      fun menu (items : menu_item list) = items

      fun readline (prompt : string,
		    menu : menu) =
	  let
	      val _ = RLG.init progname	(* Initialise readline interface *)

	      (* Build RLG completion for RLB function completions *)
	      fun fun_completion []	= RLG.NONE
		| fun_completion (h::t)	= let val options = h ()
					      val cont = fun_completion t
					  in case options of
						 []	=> RLG.STRING cont	(* No options? Allow "any string" *)
					       | _	=> RLG.ENUM (options, cont)
					  end

	      (* Build full completion table for menu *)
	      fun build_rltable_menu menu = let val items = map build_rltable_item menu
					    in RLG.ENUM_SPLIT (items,
							       RLG.NONE)
					    end
	      and build_rltable_item (FUNCTION rlfun)		= (#name rlfun, fun_completion (List.rev (#completions rlfun)))
		| build_rltable_item (SUBMENU (name, menu))	= (name, build_rltable_menu menu)

	      fun rebuild_completion_table () =
		  let
		      val rltable = build_rltable_menu menu
		      val _ = RLG.set_readline_table rltable
		  in ()
		  end

	      val _ = rebuild_completion_table ();
	  in
	      RLG.readline prompt
	  end

      fun read (prompt : string) =
	  let 
	      val _ = RLG.init progname	(* Initialise readline interface *)
	      val _ = RLG.set_readline_table RLG.NONE
	  in
	      RLG.readline prompt
	  end

      fun process (commands : string,
		   menu : menu) =
	  let
	      val tokenise = String.tokens (fn c => List.exists (fn c' => c = c') [#" ", #"\t", #"\n", #"\r"])

	      fun process commandline =
		  let val tokens = tokenise commandline

		      fun find_fun ([], _)			= if tokens = []
								  then ()
								  else print "<no action>"
			| find_fun (h::t, [])			= print ("No match: '" ^ h ^ "'")
			| find_fun (nm::nmtail,
				    (FUNCTION f)::menutail)	= if (#name f) = nm
								  then (#function f) nmtail
								  else find_fun (nm::nmtail, menutail)
			| find_fun (nm::nmtail,
				    (SUBMENU (nm', menu)
				     ::menutail))		= if nm = nm'
								  then find_fun (nmtail, menu)
								  else find_fun (nm::nmtail, menutail)
		      val _ = find_fun (tokens, menu)
		      val _ = print "\n"
		  in ()
		  end
	  in
	      (process commands)
	      handle RLB.ParseError err => (print (RLB.stringify_error err);
					    print "\n")
	  end

      fun process_readline (prompt : prompt,
			    menu : menu) =
	  let val input = readline (prompt, menu)
	  in process (input, menu)
	  end

end